﻿#if DEBUG
using System.Collections.Generic;
using NUnit.Framework;
using NUnit.Framework.SyntaxHelpers;

namespace Marimo.開発支援.コンソールアプリケーションライブラリ._テスト
{
    /// <summary>
    /// コマンドライン引数管理クラスに対するテストを行います。
    /// </summary>
    [TestFixture]
    public class コマンドライン引数管理のテスト : AssertionHelper
    {
        #region Setup/Teardown

        /// <summary>
        /// 各テストの初期処理です。
        /// </summary>
        [SetUp]
        public void SetUp()
        {
            コマンドライン引数管理.初期化する();
        }

        #endregion

        private Dictionary<string, string> 結果;

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、コマンドキーを登録した後で大文字小文字のみが異なるエイリアスを登録し、例外が発生しないことを確認します。
        /// </summary>
        [Test]
        public void エイリアスとキー登録重複_大文字小文字を区別しする_キー先()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("C");
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "c"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、コマンドキーを登録した後で同じエイリアスを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンドキーと重複するエイリアス登録),
            ExpectedMessage = "同一名称のコマンドとエイリアス「C」が登録されています。")]
        public void エイリアスとキー登録重複_大文字小文字を区別しする_キー先_エラー()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("C");
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "C"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、コマンドキーを登録した後で大文字小文字のみが異なるエイリアスを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンドキーと重複するエイリアス登録),
            ExpectedMessage = "同一名称のコマンドとエイリアス「c」が登録されています。")]
        public void エイリアスとキー登録重複_大文字小文字を区別しない_キー先()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "C"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、エイリアスを登録後に登録したエイリアスと大文字小文字のみが異なるコマンドキーを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンドキーと重複するエイリアス登録),
            ExpectedMessage = "同一名称のコマンドとエイリアス「c」が登録されています。")]
        public void エイリアスとキー登録重複_大文字小文字を区別しない_キー後()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "c"));
            コマンドライン引数管理.コマンドを登録する("C");
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、エイリアスを登録した後で大文字小文字のみが異なるコマンドキーを登録し、例外が発生しないことを確認します。
        /// </summary>
        [Test]
        public void エイリアスとキー登録重複_大文字小文字を区別する_キー後()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "C"));
            コマンドライン引数管理.コマンドを登録する("c");
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、エイリアスを登録した後で同じコマンドキーを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンドキーと重複するエイリアス登録),
            ExpectedMessage = "同一名称のコマンドとエイリアス「C」が登録されています。")]
        public void エイリアスとキー登録重複_大文字小文字を区別する_キー後_エラー()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "C"));
            コマンドライン引数管理.コマンドを登録する("C");
        }

        /// <summary>
        /// 複数のコマンドで重複したエイリアスを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.エイリアス登録重複),
            ExpectedMessage = "エイリアス「command」が、複数回登録されています。")]
        public void エイリアス登録重複_大文字小文字を区別しない_別コマンド()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "command"));
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "com", "Command"));
        }

        /// <summary>
        /// 同一のコマンドに対応するように重複したエイリアスを登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.エイリアス登録重複),
            ExpectedMessage = "エイリアス「command」が、複数回登録されています。")]
        public void エイリアス登録重複_大文字小文字を区別しない_同一コマンド()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command", "co", "command"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定した後に、複数のコマンドで重複したエイリアスを登録し、例外メッセージのエイリアス名が登録の通りに表示されていることを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.エイリアス登録重複),
            ExpectedMessage = "エイリアス「Command」が、複数回登録されています。")]
        public void エイリアス登録重複_大文字小文字を区別する_別コマンド_エラーメッセージ()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "com", "Command"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定した後に、複数のコマンドに対して大文字小文字のみが異なるエイリアスを登録し、例外が発生しないことを確認します。
        /// </summary>
        [Test]
        public void エイリアス登録重複_大文字小文字を区別する_別コマンド_エラー発生せず()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));
            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "com", "command"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定した後に、同一のコマンドに対して重複したエイリアスを登録し、例外メッセージのエイリアス名が登録の通りに表示されていることを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.エイリアス登録重複),
            ExpectedMessage = "エイリアス「Command」が、複数回登録されています。")]
        public void エイリアス登録重複_大文字小文字を区別する_同一コマンド_エラーメッセージ()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command", "co", "Command"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定した後に、同一のコマンドに対して大文字小文字のみが異なるエイリアスを登録し、例外が発生しないことを確認します。
        /// </summary>
        [Test]
        public void エイリアス登録重複_大文字小文字を区別する_同一コマンド_エラー発生せず()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command", "co", "command"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、区別されないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別しない()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/D",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("d"));
            Expect(結果["d"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、結果からコマンドを取得する場合については、
        /// 大文字小文字を区別して登録されたコマンドを指定しなければいけないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別しない_登録コマンドの呼び出しは区別する()
        {
            コマンドライン引数管理.コマンドを登録する("D");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/d",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("D"));
            Expect(結果.ContainsKey("d"), Is.False);

            Expect(結果["D"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、複数文字からなるコマンドを登録、指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別しない_複数文字コマンド()
        {
            コマンドライン引数管理.コマンドを登録する("Command");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/cOmMaNd",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("Command"));
            Expect(結果["Command"], Is.EqualTo("abcde"));

            Expect(結果.ContainsKey("cOmMaNd"), Is.False);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、登録、指定時に区別されることを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("C");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "fghij",
                                                 "/C",
                                                 "klmn"
                                             });

            Expect(結果.Count, Is.EqualTo(2));

            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.EqualTo("fghij"));

            Expect(結果.ContainsKey("C"));
            Expect(結果["C"], Is.EqualTo("klmn"));
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するプロパティの取得、設定を確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する_取得および設定()
        {
            Expect(コマンドライン引数管理.大文字小文字を区別する, Is.False);

            コマンドライン引数管理.大文字小文字を区別する = true;

            Expect(コマンドライン引数管理.大文字小文字を区別する, Is.True);

            コマンドライン引数管理.大文字小文字を区別する = false;

            Expect(コマンドライン引数管理.大文字小文字を区別する, Is.False);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するプロパティの設定をfalseからtrueに変更しても、登録されたエイリアスが変化しないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する_設定時に登録エイリアス情報維持_FalseからTrue()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);

            コマンドライン引数管理.大文字小文字を区別する = true;

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/Command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するプロパティの設定をtrueからfalseに変更しても、登録されたエイリアスが変化しないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する_設定時に登録エイリアス情報維持_TrueからFalse()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/Command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);


            コマンドライン引数管理.大文字小文字を区別する = false;

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するプロパティの設定falseからtrueも、登録されたコマンドが変化しないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する_設定時に登録キー情報維持_FalseからTrue()
        {
            コマンドライン引数管理.コマンドを登録する("C");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("C"));
            Expect(結果["C"], Is.Null);

            コマンドライン引数管理.大文字小文字を区別する = true;

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/C"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("C"));
            Expect(結果["C"], Is.Null);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するプロパティの設定をtrueからfalseに変更しても、登録されたコマンドが変化しないことを確認します。
        /// </summary>
        [Test]
        public void コマンドの大文字小文字を区別する_設定時に登録キー情報維持_TrueからFalse()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);


            コマンドライン引数管理.大文字小文字を区別する = false;

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/C"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドを一つ登録、指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド一つ指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// 解釈できないコマンドライン引数が先頭に指定された時に例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド引数不正),
            ExpectedMessage = "コマンドに対応しないコマンド引数が指定されています。指定された引数は「c」です。")]
        public void コマンド引数不整合_先頭()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "c"
                                             });
        }

        /// <summary>
        /// 解釈できないコマンドライン引数が先頭以外に指定された時に例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド引数不正),
            ExpectedMessage = "コマンドに対応しないコマンド引数が指定されています。指定された引数は「/c」、「引数」、「二つ目」です。")]
        public void コマンド引数不整合_先頭以外()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "引数",
                                                 "二つ目"
                                             });
        }

        /// <summary>
        /// 文字列のみではなく、コマンドクラスのインスタンスでコマンドを指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// エイリアスを登録してコマンドを指定し、結果としてエイリアスに対応付けられたコマンドが取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定_エイリアス()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "コマンド"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/コマンド"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、エイリアスの登録したあと、登録したエイリアスと大文字小文字のみ異なるエイリアスの指定を行い、結果としてエイリアスに対応付けられたコマンドが取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定_エイリアス_大文字小文字を区別しない()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/CoMmAnD"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、大文字小文字のみが異なるエイリアスの登録したあとエイリアスの指定を行い、大文字小文字の違いを判別して結果としてエイリアスに対応付けられたコマンドが取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定_エイリアス_大文字小文字を区別する()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "command"));
            コマンドライン引数管理.コマンドを登録する(new コマンド("C", false, "Command"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/Command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("C"));
            Expect(結果["C"], Is.Null);
        }

        /// <summary>
        /// 一つのコマンドに対して複数のエイリアスを登録してそのコマンドを指定し、結果としてエイリアスに対応付けられたコマンドが取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定_エイリアス_複数()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "コマンド", "Command"));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/Command"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// 指定が任意であるコマンドを登録し、指定しなくても結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド指定_任意()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false));
            コマンドライン引数管理.コマンドを登録する(new コマンド("d", false));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// 指定が必須であるコマンドを登録し、指定しなければ例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.必須コマンド不足),
            ExpectedMessage = "必須コマンド「/d」が指定されていません。")]
        public void コマンド指定_必須()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", true));
            コマンドライン引数管理.コマンドを登録する(new コマンド("d", true));

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c"
                                             });
        }


        /// <summary>
        /// 複数のコマンドを登録、指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void コマンド複数指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("d");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "/d"
                                             });

            Expect(結果.Count, Is.EqualTo(2));

            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);

            Expect(結果.ContainsKey("d"));
            Expect(結果["d"], Is.Null);
        }

        /// <summary>
        /// 先頭がハイフンとなるコマンドを一つ指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void ハイフンコマンド一つ指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "-c"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.Null);
        }

        /// <summary>
        /// コマンドに対する引数となる値を指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 値つきコマンド一つ指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// 複数のコマンドに対する引数となる値を指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 値つきコマンド複数指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("d");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "fghij",
                                                 "/d",
                                                 "klmn"
                                             });

            Expect(結果.Count, Is.EqualTo(2));

            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.EqualTo("fghij"));

            Expect(結果.ContainsKey("d"));
            Expect(結果["d"], Is.EqualTo("klmn"));
        }

        /// <summary>
        /// 先頭がハイフンのコマンドに、コマンドに対する引数となる値を指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 値つきハイフンコマンド一つ指定()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "-d",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("d"));
            Expect(結果["d"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// コマンドに対する引数となる値を持つコマンドと、持たないコマンドを同時に指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 値のつくものとつかないものを混合したコマンド複数指定()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("d");
            コマンドライン引数管理.コマンドを登録する("e");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/c",
                                                 "fghij",
                                                 "/d",
                                                 "/e",
                                                 "klmn"
                                             });

            Expect(結果.Count, Is.EqualTo(3));

            Expect(結果.ContainsKey("c"));
            Expect(結果["c"], Is.EqualTo("fghij"));

            Expect(結果.ContainsKey("d"));
            Expect(結果["d"], Is.Null);

            Expect(結果.ContainsKey("e"));
            Expect(結果["e"], Is.EqualTo("klmn"));
        }

        /// <summary>
        /// 初期化するメソッドを呼び出し、大文字小文字を区別する情報が初期化されることを確認します。
        /// </summary>
        [Test]
        public void 初期化する_大文字小文字を区別する()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("D");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/D"
                                             });

            Expect(結果.Count, Is.EqualTo(1));

            Expect(結果.ContainsKey("D"));
            Expect(結果.ContainsKey("d"), Is.False);

            コマンドライン引数管理.初期化する();

            コマンドライン引数管理.コマンドを登録する("D");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/d"
                                             });

            Expect(結果.Count, Is.EqualTo(1));

            Expect(結果.ContainsKey("D"));
            Expect(結果.ContainsKey("d"), Is.False);
        }

        /// <summary>
        /// 初期化を行い、エイリアス情報がクリアされていることを確認なする。
        /// </summary>
        [Test]
        public void 初期化する_登録エイリアス情報()
        {
            コマンドライン引数管理.コマンドを登録する(new コマンド("c", false, "Command"));

            コマンドライン引数管理.初期化する();

            コマンドライン引数管理.コマンドを登録する(new コマンド("co", false, "Command"));
        }

        /// <summary>
        /// 初期化を行い、登録キー情報がクリアされていることを確認する。
        /// </summary>
        [Test]
        public void 初期化する_登録キー情報()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            コマンドライン引数管理.初期化する();

            コマンドライン引数管理.コマンドを登録する("c");
        }

        /// <summary>
        /// 同じコマンドを重複して指定し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド指定重複),
            ExpectedMessage = "コマンド「c」が、複数回指定されています。")]
        public void 指定重複()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "-c",
                                            "-c"
                                        });
        }

        /// <summary>
        /// 先頭のハイフンとスラッシュしか違わないコマンドを指定して例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド指定重複),
            ExpectedMessage = "コマンド「c」が、複数回指定されています。")]
        public void 指定重複_先頭文字が違う()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "-c",
                                            "/c"
                                        });
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、大文字小文字しか違わないコマンドを指定して例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド指定重複),
            ExpectedMessage = "コマンド「c」が、複数回指定されています。")]
        public void 指定重複_大文字小文字を区別しない()
        {
            コマンドライン引数管理.コマンドを登録する("c");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "-c",
                                            "-C"
                                        });
        }

        /// <summary>
        /// コマンドが登録されていない状態でコマンドを指定し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.未登録コマンド呼び出し),
            ExpectedMessage = "コマンド「/c」には対応していません。")]
        public void 登録していないコマンド()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "/c"
                                        });
        }

        /// <summary>
        /// コマンドが登録されていない状態でコマンドを指定し、例外メッセージのコマンド名が指定したコマンド名になっていることを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.未登録コマンド呼び出し),
            ExpectedMessage = "コマンド「/e」には対応していません。")]
        public void 登録していないコマンド_コマンド名()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "/e"
                                        });
        }

        /// <summary>
        /// コマンドが登録されていない状態でハイフンで始まるコマンドを指定し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.未登録コマンド呼び出し),
            ExpectedMessage = "コマンド「-c」には対応していません。")]
        public void 登録していないコマンド_ハイフン()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "-c"
                                        });
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、コマンドが登録されていない状態でコマンドを指定し、例外メッセージのコマンド名が指定したコマンド名になっていることを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.未登録コマンド呼び出し),
            ExpectedMessage = "コマンド「/Command」には対応していません。")]
        public void 登録していないコマンド_大文字小文字を区別してメッセージ()
        {
            コマンドライン引数管理.コマンドを登録する("d");

            コマンドライン引数管理.コマンド一覧を取得する(new[]
                                        {
                                            "/Command"
                                        });
        }

        /// <summary>
        /// 同じコマンドを重複して登録し、例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド登録重複),
            ExpectedMessage = "コマンド「c」が、複数回登録されています。")]
        public void 登録重複()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("c");
        }

        /// <summary>
        /// コマンドの大文字小文字を区別しないよう設定し、大文字小文字しか違わないコマンドを登録して例外が発生することを確認します。
        /// </summary>
        [Test]
        [ExpectedException(
            ExceptionType = typeof (例外管理.コマンド登録重複),
            ExpectedMessage = "コマンド「C」が、複数回登録されています。")]
        public void 登録重複_大文字小文字を区別しない()
        {
            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("C");
        }

        /// <summary>
        /// コマンドの大文字小文字を区別するよう設定し、大文字小文字しか違わないコマンドを登録して例外が返らないことを確認します。
        /// </summary>
        [Test]
        public void 登録重複_大文字小文字を区別する()
        {
            コマンドライン引数管理.大文字小文字を区別する = true;

            コマンドライン引数管理.コマンドを登録する("c");
            コマンドライン引数管理.コマンドを登録する("C");
        }

        /// <summary>
        /// 複数文字からなるコマンドを登録、指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 複数文字コマンド()
        {
            コマンドライン引数管理.コマンドを登録する("command");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/command",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("command"));
            Expect(結果["command"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// 非アスキー文字からなるコマンドを登録、指定し、結果が取得できることを確認します。
        /// </summary>
        [Test]
        public void 非アスキー文字コマンド()
        {
            コマンドライン引数管理.コマンドを登録する("日本語");

            結果 = コマンドライン引数管理.コマンド一覧を取得する(new[]
                                             {
                                                 "/日本語",
                                                 "abcde"
                                             });

            Expect(結果.Count, Is.EqualTo(1));
            Expect(結果.ContainsKey("日本語"));
            Expect(結果["日本語"], Is.EqualTo("abcde"));
        }

        /// <summary>
        /// 説明プロパティの取得、設定ができることを確認します。
        /// </summary>
        [Test]
        public void 説明()
        {
            コマンドライン引数管理.説明 = "説明文";

            Expect(コマンドライン引数管理.説明, Is.EqualTo("説明文"));
        }

        /// <summary>
        /// 初期化するメソッドにより説明プロパティが初期化することを確認します。
        /// </summary>
        [Test]
        public void 初期化する_説明()
        {
            コマンドライン引数管理.説明 = "説明文";

            コマンドライン引数管理.初期化する();
            Expect(コマンドライン引数管理.説明, Is.Empty);
        }

    }
}
#endif